## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(collapse = TRUE, comment = "#>", fig.align = "center")
library(ggplot2)
library(sf)
library(insetplot)
library(patchwork)
# Shared data available to all chunks
nc <- st_read(system.file("shape/nc.shp", package = "sf"), quiet = TRUE)

## ----quick-start, out.width="100%",fig.asp=0.4,dpi=300, fig.alt="North Carolina map with an inset showing a zoomed detail region at the bottom-left corner."----
# 1) Configure the layout: one main map + one inset (nc loaded in setup)
config_insetmap(
    data_list = list(nc),
    specs = list(
        inset_spec(main = TRUE),
        inset_spec(
            xmin = -82, xmax = -80.5, ymin = 35.5, ymax = 36, # bbox for the inset area
            loc = "left bottom", # where to place the inset
            scale_factor = 2 # size relative to main
        )
    )
)

# 2) Build a base plot (shared by main and inset unless a spec supplies its own)
base <- ggplot(nc, aes(fill = AREA)) +
    geom_sf() +
    scale_fill_viridis_c() +
    guides(fill = "none") +
    theme_void()

# 3) Compose
with_inset(base)

# 4) Save with correct aspect ratio (only width or height is needed)
# ggsave_inset("inset_map.png", width = 10)

## ----quick-start-custom, out.width="100%",fig.asp=0.4,dpi=300, fig.alt="North Carolina map with an inset showing a zoomed detail region labeled 'Detail' at the bottom-left corner."----
main <- ggplot(nc, aes(fill = AREA)) +
    geom_sf() +
    scale_fill_viridis_c() +
    guides(fill = "none") +
    theme_void()

config_insetmap(
    data_list = list(nc),
    specs = list(
        inset_spec(main = TRUE, plot = main),
        inset_spec(
            xmin = -82, xmax = -80.5, ymin = 35.5, ymax = 36,
            loc = "left bottom", # where to place the inset
            scale_factor = 2, # size relative to main
            plot = main +
                annotate("label", x = -80.5, y = 35.5, label = "Detail", size = 5, hjust = 1, vjust = 0, fill = "white", size.unit = "pt")
        )
    )
)
with_inset()

## ----realmap-setup------------------------------------------------------------
inset_bbox <- c(xmin = -82, xmax = -80.5, ymin = 35.5, ymax = 36)
main_map <- ggplot(nc, aes(fill = AREA)) +
    geom_sf() +
    scale_fill_viridis_c() +
    guides(fill = "none") +
    theme_void() +
    map_border(linewidth = 0.5)
inset_map <- main_map +
    coord_sf(
        xlim = c(inset_bbox["xmin"], inset_bbox["xmax"]),
        ylim = c(inset_bbox["ymin"], inset_bbox["ymax"])
    )

## ----compare-patchwork-real, fig.cap='Inset via patchwork', out.width="100%",fig.asp=0.4,dpi=300, fig.alt="North Carolina map with an inset positioned at bottom-left using patchwork, showing some visual distortion due to mismatched aspect ratio."----
# Manual insetting with patchwork: force a mismatched container ratio
main_map + patchwork::inset_element(
    inset_map,
    left = 0, bottom = 0, right = 0.3, top = 0.3, # arbitrary width/height
    align_to = "panel"
)

## ----compare-cowplot-real, fig.cap='Inset via cowplot', out.width="100%",fig.asp=0.4,dpi=300, fig.alt="North Carolina map with an inset positioned at bottom-left using cowplot with 30% width and height, showing some distortion."----
cowplot::ggdraw(main_map) +
    cowplot::draw_plot(inset_map, 0, 0, 0.3, 0.3) # arbitrary width/height

## ----compare-cowplot-real-asp, fig.cap='Inset via cowplot with wrong aspect ratio', out.width="100%",fig.asp=1.0,dpi=300, fig.alt="North Carolina map with an inset using cowplot with square aspect ratio (1:1), showing significant distortion due to incompatible aspect ratio."----
cowplot::ggdraw(main_map) +
    cowplot::draw_plot(inset_map, 0, 0, 0.3, 0.3) # arbitrary width/height

## ----compare-insetplot-real, fig.cap='Inset via insetplot', out.width="100%",fig.asp=0.4,dpi=300, fig.alt="North Carolina map with an inset positioned at bottom-left using insetplot, showing correct aspect ratio preservation without distortion."----
config_insetmap(
    data_list = list(nc),
    specs = list(
        inset_spec(main = TRUE),
        inset_spec(
            xmin = -82, xmax = -80.5, ymin = 35.5, ymax = 36,
            loc = "left bottom", height = 0.3
        )
    )
)

# You even don't need to prepare `inset_map` here since insetplot handles it internally
with_inset(main_map)

## ----compare-insetplot-real-asp, fig.cap='Inset via insetplot with wrong aspect ratio', out.width="100%",fig.asp=1.0,dpi=300, fig.alt="North Carolina map with an inset using insetplot with square aspect ratio (1:1), demonstrating that insetplot still maintains correct spatial aspect despite the output canvas ratio mismatch."----
with_inset(main_map)

## ----params-inset-spec, results='hide', message=FALSE, warning=FALSE----------
# Define a main spec (no size/position needed)
inset_spec(main = TRUE)

# Define an inset by bbox + position + size
inset_spec(
    xmin = -120, xmax = -100, ymin = 30, ymax = 50, # spatial extent (data CRS)
    loc = "right bottom", # shorthand position on full canvas
    width = 0.30 # size in [0,1]; prefer ONE of width/height
)

# Prefer scale_factor to size relative to main plot automatically
inset_spec(
    xmin = -120, xmax = -100, ymin = 30, ymax = 50,
    loc = "left bottom",
    scale_factor = 0.5 # relative to main ranges
)

## ----params-config-insetmap, results='hide', message=FALSE, warning=FALSE-----
cfg <- config_insetmap(
    data_list = list(nc), # list of sf objects used to compute extents/CRS
    specs = list(
        inset_spec(main = TRUE),
        inset_spec(xmin = -84, xmax = -75, ymin = 33, ymax = 37, loc = "left bottom", scale_factor = 0.5)
    ),
    crs = sf::st_crs("EPSG:4326"), # target CRS for coord_sf()
    border_args = list(color = "black", linewidth = 1) # passed to map_border()
)

## ----params-with-inset--------------------------------------------------------
out <- with_inset(base) # returns the composed plot
names(with_inset(base, .return_details = TRUE)) # list with full, subplots, layouts, main_ratio

# You can also provide custom plots after configuring
out <- with_inset(list(main_map, inset_map))

## ----params-ggsave-inset, eval=FALSE------------------------------------------
#  # Provide only width (or only height). The other dimension is computed to match the main ratio.
#  # ggsave_inset("map.png", p, width = 10)

